#region Copyright
// -------------------------------
// Copyright  2007 Andreas Brsen
// -------------------------------
#endregion

using System;
using System.Resources;
using BRuKE.Article.UseLegacyCppFromCSharp.ManagedCodeInterfaces;
using BRuKE.Article.UseLegacyCppFromCSharp.MixedModeCode;

namespace BRuKE.Article.UseLegacyCppFromCSharp.ManagedCode
{
   /// <summary>
   /// Class, which provides a way to access the data, which are delivered from the managed world 
   /// to the legacy world, which calculates the given values and returns the result back to the 
   /// managed world.
   /// </summary>
   public class LegacyCodeFacade
   {
      #region member variables

      private ILegacyCodeAdapter m_ILegacyCodeAdpater;
      private ManagedDelegateContainer m_ManagedDelegateContainer;

      #endregion

      /// <summary>
      /// Constructor.
      /// </summary>
      public LegacyCodeFacade()
      {
         m_ILegacyCodeAdpater = new LegacyCodeAdapter();

         m_ManagedDelegateContainer = new ManagedDelegateContainer();
         m_ManagedDelegateContainer.LoadStringOutOfResourceManagedDelegate =
            new LoadStringOutOfResourceManagedDelegate( LoadStringOutOfResource );

         m_ManagedDelegateContainer.DoSomethingWithAEnumValueManagedDelegate =
            new DoSomethingWithAEnumValueManagedDelegate( DoSomethingWithAEnumValue );
      }

      #region delegate methods

      /// <summary>
      /// This delegate delivers for a given unmanaged enum the appropriate long value.
      /// </summary>
      /// <param name="managedEnum">
      /// The managed enum value.
      /// </param>
      /// <returns>
      /// The appropriate long value for the given enum.
      /// </returns>
      /// <exception cref="MixedModeCodeException">
      /// If the return value could not be retrieved.
      /// </exception>
      private int DoSomethingWithAEnumValue( ManagedEnum managedEnum )
      {
         int returnValue;
         returnValue = -1;

         if( managedEnum == ManagedEnum.ValueOne )
         {
            returnValue = 123;
         }
         else
         if( managedEnum == ManagedEnum.ValueTwo )
         {
            returnValue = 321;
         }

         return ( returnValue );
      }

      /// <summary>
      /// Get a string for the legacy code out of the resources, which are placed in the 
      /// managed .NET managed code.
      /// </summary>
      /// <param name="resourceId">
      /// The resource Id.
      /// </param>
      /// <param name="resourceExistsReturnValue">
      /// The value, which indicate, whether a string exists for the given resource 
      /// ID (!=0) or not (0)
      /// </param>
      /// <returns>
      /// The found string or null, if no string exists for the given resource Id.
      /// </returns>
      /// <exception cref="MixedModeCodeException">
      /// If the resource could not be read.
      /// </exception>
      private string LoadStringOutOfResource( int resourceId, ref int resourceExistsReturnValue )
      {
         string returnValue;

         try
         {
            // Read for the given resource Id the appropriate string out of the resources.
            // HINT: THe rule is, that the legacy resources are prefixed with the "LegacyResource_"
            // string, so that they can be distinguished between the resources for the managed code.
            // -------------------------------------------------------------------------------------
            returnValue = 
               Resources.ResourceManager.GetString( "LegacyResource_" + resourceId.ToString() );

            if( returnValue != null )
            {
               resourceExistsReturnValue = 1;
            }
            else
            {
               resourceExistsReturnValue = 0;
            }
         }
         catch( InvalidOperationException ex )
         {
            throw new MixedModeCodeException
            ( 
               string
                  .Format
                  ( 
                     "The resource for the given resource Id {0} is not a string in the managed world.", 
                     resourceId 
                  ),
                  ex 
            );
         }
         catch( MissingManifestResourceException ex )
         {
            throw new MixedModeCodeException
            ( 
               string
                  .Format
                  ( 
                     "The manifest in which the given resource Id {0} is missing in the managed world.", 
                     resourceId 
                  ),
                  ex 
            );
         }

         return( returnValue );
      }

      #endregion

      #region public methods

      /// <summary>
      /// Activates the legacy code functionality.
      /// </summary>
      /// <returns>
      /// The string, which is created by the legacy code.
      /// </returns>
      public string GetLegacyInformation()
      {
         string returnValue;
         returnValue = m_ILegacyCodeAdpater.StartLegacyCode( m_ManagedDelegateContainer );
         return ( returnValue );
      }

      #endregion
   }
}
