// -------------------------------
// Copyright  2007 Andreas Brsen
// -------------------------------

#include <windows.h>

#using <mscorlib.dll>

using namespace System;

#include "LegacyCodeAdapter.h"
#include "LegacyCodeAdapterDelegates.h"

using namespace BRuKE::Article::UseLegacyCppFromCSharp::MixedModeCode;

/// <summary>
/// Default constructor.
/// </summary>
LegacyCodeAdapter::LegacyCodeAdapter()
{
   // Do nothing.
}

/// <summary>
/// Destructor.
/// </summary>
LegacyCodeAdapter::~LegacyCodeAdapter()
{
}

/// <summary>
/// Method to start the lagacy code.
/// </summary>
/// <param name="managedDelegateContainer">
/// The container, which contains all delegates, which are used inside the 
/// legacy code to communicate with the outside .NET managed world.
/// </param>
/// <returns>
/// The output of the legacy code. This output filled up by the managed delegate methods.
/// </returns>
/// <exception cref="ArgumentNullException">
/// If the given delegate container is null.
/// </exception>
String^ LegacyCodeAdapter::StartLegacyCode
( 
   ManagedDelegateContainer^ managedDelegateContainer
)
{
   // Check the arguements.
   // ---------------------
   if( managedDelegateContainer == nullptr )
   {
      throw gcnew ArgumentNullException( "managedDelegateContainer" );
   }

   String^ returnValue;
   returnValue = nullptr;

   GCHandle gcHandleLoadStringOutOfResourceUnmanagedDelegate;
   GCHandle gcHandleDoSomethingWithAEnumUnmanagedDelegate;
   LegacyCodeCallbackContainer* pLegacyCodeCallbackContainer;
   pLegacyCodeCallbackContainer = nullptr;

   try
   {
      // Store the given managed delegate container in a member variable, so that the delegeates 
      // inside of the container can be used in the converter methods.
      m_ManagedDelegateContainer = managedDelegateContainer;
      
      // Assign converter methods to the mixed mode delegates, to be able to use them inside the 
      // Mixed Mode legacy code. 
      // ---------------------------------------------------------------------------------------
      LoadStringOutOfResourceMixedModeDelegate^ loadStringOutOfResourceMixedModeDelegate;
      loadStringOutOfResourceMixedModeDelegate =
         gcnew LoadStringOutOfResourceMixedModeDelegate
         ( 
            this, 
            &LegacyCodeAdapter::LoadStringOutOfResourceConverter 
         );

      DoSomethingWithAEnumValueMixedModeDelegate^ doSomethingWithAEnumValueMixedModeDelegate;
      doSomethingWithAEnumValueMixedModeDelegate = 
         gcnew DoSomethingWithAEnumValueMixedModeDelegate    
         (
            this,
            &LegacyCodeAdapter::DoSomethingWithAEnumValueConverter
         );

      //----------------------------------------------------------
      // Declare the GCHandle so that objects out of managed code 
      // can be accessed from unmanged code. The created handle 
      // is of type GCHandleType.Normal. This prevent the callback 
      // method to be deallocated by the garbage collector.
      //----------------------------------------------------------
      gcHandleLoadStringOutOfResourceUnmanagedDelegate = 
         GCHandle::Alloc( loadStringOutOfResourceMixedModeDelegate );

      gcHandleDoSomethingWithAEnumUnmanagedDelegate = 
         GCHandle::Alloc( doSomethingWithAEnumValueMixedModeDelegate );

      // -------------------------------------------------------
      // Get back the function pointer to the callback function.
      // -------------------------------------------------------
      IntPtr intPtrLoadStringOutOfResourceUnmanagedDelegate;
      IntPtr intPtrDoSomethingWithAEnumUnmanagedDelegate;

      intPtrLoadStringOutOfResourceUnmanagedDelegate = 
         Marshal::GetFunctionPointerForDelegate( loadStringOutOfResourceMixedModeDelegate );

      intPtrDoSomethingWithAEnumUnmanagedDelegate = 
         Marshal::GetFunctionPointerForDelegate( doSomethingWithAEnumValueMixedModeDelegate );

      // Create the callback container, which transports all callbacks into the legacy world.
      pLegacyCodeCallbackContainer = new LegacyCodeCallbackContainer();

      // -------------------------------------------------------
      // Assign the existing callback methods, so that they can 
      // be transported via the callback container.
      // -------------------------------------------------------
      pLegacyCodeCallbackContainer->LoadStringOutOfResourceUnmanagedCallback = 
         static_cast<LegacyCodeCallbacks::LoadStringOutOfResourceUnmanagedCallback>
         (
            intPtrLoadStringOutOfResourceUnmanagedDelegate.ToPointer()
         );

      pLegacyCodeCallbackContainer->DoSomethingWithAEnumValueUnmanagedCallback = 
         static_cast<LegacyCodeCallbacks::DoSomethingWithAEnumValueUnmanagedCallback>
         (
            intPtrDoSomethingWithAEnumUnmanagedDelegate.ToPointer()
         );

      // Start of the legacy code functionality.
      // ---------------------------------------
      LegacyCode legacyCode;
      LPTSTR legacyCodeReturnString;

      legacyCodeReturnString = legacyCode.StartLegacyCode( pLegacyCodeCallbackContainer );

      returnValue = gcnew String( legacyCodeReturnString );

      delete legacyCodeReturnString;
      legacyCodeReturnString = nullptr;
   }
   finally
   {
      IfGcHandleAllocatedFreeThem( gcHandleLoadStringOutOfResourceUnmanagedDelegate );
      IfGcHandleAllocatedFreeThem( gcHandleDoSomethingWithAEnumUnmanagedDelegate );

      // Delete the allocated resources.
      // -------------------------------      
      if( pLegacyCodeCallbackContainer != nullptr )
      {
         delete pLegacyCodeCallbackContainer;
      }
   }

   return( returnValue );
}

// ------------------
// Converter methods.
// ------------------

/// <summary>
/// Get a string for the legacy code out of the resources, which are placed in the 
/// managed .NET managed code.
/// </summary>
/// <param name="resourceId">
/// The resource Id.
/// </param>
/// <param name="resourceExistsReturnValue">
/// The value, which indicate, whether a string exists for the given resource 
/// ID (!=0) or not (0)
/// </param>
/// <returns>
/// The found string or null, if no string exists for the given resource Id.
/// </returns>
/// <exception cref="MixedModeCodeException">
/// If the resource could not be read.
/// </exception>
LPTSTR LegacyCodeAdapter::LoadStringOutOfResourceConverter
(
   long resourceId,
   int& resourceExistsReturnValue
)
{
   String^ callbackReturnValue;
   callbackReturnValue = 
      m_ManagedDelegateContainer
         ->LoadStringOutOfResourceManagedDelegate( resourceId, resourceExistsReturnValue );

   LPTSTR returnValue;

   if( resourceExistsReturnValue != 0 )
   {
     returnValue = (LPTSTR)Marshal::StringToCoTaskMemAnsi( callbackReturnValue ).ToPointer();
   }

   return( returnValue ); 
}

/// <summary>
/// This delegate delivers for a given unmanaged enum the appropriate long value.
/// </summary>
/// <param name="lagacyCodeEnum">
/// The managed enum value.
/// </param>
/// <returns>
/// The appropriate long value for the given enum.
/// </returns>
/// <exception cref="MixedModeCodeException">
/// If the return value could not be retrieved.
/// </exception>
long LegacyCodeAdapter::DoSomethingWithAEnumValueConverter
( 
   Legacy::LegacyCodeEnum lagacyCodeEnum 
)
{
   long returnValue;

   // Convert the given enum value to the corresponding enum out of the managed world.
   ManagedCodeInterfaces::ManagedEnum managedEnum;
   managedEnum = 
      (ManagedCodeInterfaces::ManagedEnum)Enum::ToObject
      ( 
         ManagedCodeInterfaces::ManagedEnum::typeid,
         (int)lagacyCodeEnum
      );

   returnValue = m_ManagedDelegateContainer->DoSomethingWithAEnumValueManagedDelegate( managedEnum );

   return( returnValue );
}

// <summary>
// The method, which controls the given GCHandle, whether it is allocated
// and it yes, the allocated memory is released.
// </summary>
// <params name="gcHandle">
// The GCHandle to handle.
// </params>
void LegacyCodeAdapter::IfGcHandleAllocatedFreeThem( GCHandle gcHandle )
{
    if( gcHandle.IsAllocated == true )
    {
        gcHandle.Free();
    }
}
